#!/bin/bash

# usbmemformat.sh
#
# Usage: usbmemformat.sh <device file> <filesystem> <log file> <label>
#
# device file: The device file for the usb flash memory drive, usually given as /dev/sda1
# filesystem: The integer representing the filesystem type, either 1 for vfat, 4 for ext2,
#             or 5 for ext3 
# log file: The log file for recording the output of the usb flash memory drive format script
# label: The optionally passed label string used to format the usb flash memory drive with a label
#
# Return Codes:
# 1 - Error formating the usb flash memory drive
#
# Module History
#  00  10/07/2004  P. Provost  Initial release

DEVICEFILE=$1
FILESYSTEM=$2
LOGFILE=$3
LABEL=$4

echo "-> usbmemformat.sh" >> $LOGFILE
echo "Formatting usb flash memory drive at $DEVICEFILE for `date`" >> $LOGFILE

actzTrace "XMDA   T: -> usbmemformat.sh"
actzTrace "XMDA   F: usbmemformat.sh: Formatting the usb flash memory drive at $DEVICEFILE for `date`" 

# Determine what filesystem is to be used to format the usb flash memory drive
case "$FILESYSTEM" in
    # If the usb flash memory drive is to be formatted with the vfat filesystem 
    1)
        # Check to see if the label argument exists
        if [ -z $LABEL ]
        then
            # Format the usb flash memory drive in vfat with no label
            echo "Formatting the usb flash memory drive in vfat with no label." >> $LOGFILE
            actzTrace "XMDA   F: usbmemformat.sh: Formatting the usb flash memory drive in vfat with no label."
            
            if ! /sbin/mkdosfs -F 32 -v $DEVICEFILE >> $LOGFILE 2>&1; then
                echo "Couldn't format the usb flash memory drive...exiting." >> $LOGFILE
                actzTrace "XMDA   F: usbmemformat.sh: Couldn't format a usb flash memory drive...exiting." 
                
                exit 1
            fi
        else
            # Format the usb flash memory drive in vfat with a label
            echo "Formatting the usb flash memory drive in vfat with the label: $LABEL" >> $LOGFILE
            actzTrace "XMDA   F: usbmemformat.sh: Formatting the usb flash memory drive in vfat with the label: $LABEL"
            
            if ! /sbin/mkdosfs -F 32 -v -n $LABEL $DEVICEFILE >> $LOGFILE 2>&1; then
                echo "Couldn't format the usb flash memory drive with a label...exiting." >> $LOGFILE
                actzTrace "XMDA   F: usbmemformat.sh: Couldn't format a usb flash memory drive in vfat with a label...exiting."
                
                exit 1
            fi
        fi
    ;;
    # If the usb flash memory drive is to be formatted with the ext2 filesystem
    4)
        # Check to see if the label argument exists
        if [ -z $LABEL ]
        then
            # Format the usb flash memory drive in ext2 with no label
            echo "Formatting the usb flash memory drive in ext2 with no label." >> $LOGFILE
            actzTrace "XMDA   F: usbmemformat.sh: Formatting the usb memory drive in ext2 with no label."
            
            if ! /sbin/mkfs -t ext2 -v $DEVICEFILE >> $LOGFILE 2>&1; then
                echo "Couldn't format the usb flash memory drive in ext2...exiting." >> $LOGFILE
                actzTrace "XMDA   F: usbmemformat.sh: Couldn't format the usb memory drive in ext2...exiting."
                
                exit 1
            fi
        else
            # Format the usb flash memory drive in ext2 with a label
            echo "Formatting the usb flash memory drive in ext2 with the label: $LABEL" >> $LOGFILE
            actzTrace "XMDA   F: usbmemformat.sh: Formatting the usb flash memory drive in ext2 with the label: $LABEL" 
            
            if ! /sbin/mkfs -t ext2 -v -L $LABEL $DEVICEFILE >> $LOGFILE 2>&1; then
                echo "Couldn't format the usb flash memory drive in ext2 with a label...exiting." >> $LOGFILE
                actzTrace "XMDA   F: usbmemformat.sh: Couldn't format the usb flash memory drive in ext2 with a label...exiting."
                
                exit 1
            fi
        fi
    ;;
    # If the usb flash memory drive is to be formatted with the ext3 filesystem
    5)
        # Check to see if the label argument exists
        if [ -z $LABEL ]
        then
            # Format the usb flash memory drive in ext3 with no label
            echo "Formatting the usb flash memory drive in ext3 with no label." >> $LOGFILE
            actzTrace "XMDA   F: usbmemformat.sh: Formatting the usb flash memory drive in ext3 with no label."
            
            if ! /sbin/mkfs -t ext3 -v $DEVICEFILE >> $LOGFILE 2>&1; then
                echo "Couldn't format the usb flash memory drive in ext3...exiting." >> $LOGFILE
                actzTrace "XMDA   F: usbmemformat.sh: Couldn't format the usb flash memory drive in ext3...exiting."
                
                exit 1
            fi
        else
            # Format the usb flash memory drive in ext3 with a label
            echo "Formatting the usb flash memory drive in ext3 with the label: $LABEL" >> $LOGFILE
            actzTrace "XMDA   F: usbmemformat.sh: Formatting the usb flash memory drive in ext3 with the label: $LABEL" 
            
            if ! /sbin/mkfs -t ext3 -v -L $LABEL $DEVICEFILE >> $LOGFILE 2>&1; then
                echo "Couldn't format the usb flash memory drive in ext3 with a label...exiting." >> $LOGFILE
                actzTrace "XMDA   F: usbmemformat.sh: Couldn't format the usb flash memory drive in ext3 with a label...exiting."
                
                exit 1
            fi
        fi
    ;;
esac

actzTrace "XMDA   T: <- usbmemformat.sh"

echo "<- usbmemformat.sh" >> $LOGFILE
echo "" >> $LOGFILE

exit 0

